#!/bin/sh
#
# Copyright (C) 2012-2024 Promark, Inc.
#
# Script Version for S390
#---------------------------------------------------------------------
#
#                      Clam AntiVirus: Database Updater 1.0.7
#           By The ClamAV Team: https://www.clamav.net/about.html#credits
#           (C) 2022 Cisco Systems, Inc.
#
#    freshclam [options]
#
#    --help               -h              Show this help
#    --version            -V              Print version number and exit
#    --verbose            -v              Be verbose
#    --debug                              Enable debug messages
#    --quiet                              Only output error messages
#    --no-warnings                        Don't print and log warnings
#    --stdout                             Write to stdout instead of stderr. Does not affect 'debug' messages.
#    --show-progress                      Show download progress percentage
#
#    --config-file=FILE                   Read configuration from FILE.
#    --log=FILE           -l FILE         Log into FILE
#    --daemon             -d              Run in daemon mode
#    --pid=FILE           -p FILE         Save daemon's pid in FILE
#    --foreground         -F              Don't fork into background (for use in daemon mode).
#    --user=USER          -u USER         Run as USER
#    --no-dns                             Force old non-DNS verification method
#    --checks=#n          -c #n           Number of checks per day, 1 <= n <= 50
#    --datadir=DIRECTORY                  Download new databases into DIRECTORY
#    --daemon-notify[=/path/clamd.conf]   Send RELOAD command to clamd
#    --local-address=IP   -a IP           Bind to IP for HTTP downloads
#    --on-update-execute=COMMAND          Execute COMMAND after successful update.
#                                         Use EXIT_1 to return 1 after successful database update.
#    --on-error-execute=COMMAND           Execute COMMAND if errors occurred
#    --on-outdated-execute=COMMAND        Execute COMMAND when software is outdated
#    --update-db=DBNAME                   Only update database DBNAME
#
#---------------------------------------------------------------------
# freshclam.conf default
#
#DatabaseDirectory /var/lib/clamav
#UpdateLogFile /var/log/freshclam.log
#LogFileMaxSize 2M
#LogTime yes
#LogVerbose yes
#LogSyslog yes
#LogFacility LOG_MAIL
#LogRotate yes
#PidFile /var/run/freshclam.pid
#DatabaseOwner clamav
#DNSDatabaseInfo current.cvd.clamav.net
#DatabaseMirror database.clamav.net
#MaxAttempts 5
#ScriptedUpdates yes
#CompressLocalDatabase no
#DatabaseCustomURL http://myserver.example.com/mysigs.ndb
#DatabaseCustomURL https://myserver.example.com/mysigs.ndb
#DatabaseCustomURL https://myserver.example.com:4567/allow_list.wdb
#DatabaseCustomURL ftp://myserver.example.com/example.ldb
#DatabaseCustomURL ftps://myserver.example.com:4567/example.ndb
#DatabaseCustomURL file:///mnt/nfs/local.hdb
#PrivateMirror mirror1.example.com
#PrivateMirror mirror2.example.com
#Checks 24
#HTTPProxyServer https://proxy.example.com
#HTTPProxyPort 1234
#HTTPProxyUsername myusername
#HTTPProxyPassword mypass
#HTTPUserAgent SomeUserAgentIdString
#LocalIPAddress aaa.bbb.ccc.ddd
#NotifyClamd /path/to/clamd.conf
#OnUpdateExecute command
#OnErrorExecute command
#OnOutdatedExecute command
#Foreground yes
#Debug yes
#ConnectTimeout 60
#ReceiveTimeout 300
#TestDatabases no
#Bytecode no
#ExtraDatabase dbname1
#ExtraDatabase dbname2
#ExcludeDatabase dbname1
#ExcludeDatabase dbname2
#
#---------------------------------------------------------------------
VER=18
MODE=normal

OS=`uname | env LANG=C LC_ALL=C LC_CTYPE=C tr '[:upper:]' '[:lower:]'`
case "${OS}" in
	linux)
		PROSCAN_EXEC="/etc/init.d/proscan"
		PROSCAN_CONFIG="/etc/opt/proscan/proscan.conf"
		CLASS="s390"
		MD5SUM="md5sum"
		PATH=/usr/bin:/usr/sbin:/usr/ucb:/sbin:/usr/local/bin
		;;
	*)
		echo 'Platform not supported'
		exit 2
		;;
esac

# sub
conv_facility()
{
	case "$1" in
	    mark)
		FACILITY="INTERNAL_MARK"
		;;
	    security)
		FACILITY="LOG_AUTH"
		;;
	    *)
		NAME=`echo $1 | env LANG=C LC_ALL=C LC_CTYPE=C tr '[:lower:]' '[:upper:]'`
		FACILITY="LOG_$NAME"
		;;
	esac
}

# main
CLAMAV_LIB='/usr/lib/clamav'
UPDATER='/usr/lib/clamav/freshclam'
UPDATER_CONF='/etc/clamav/freshclam.conf'
AVENGINE='/usr/lib/clamav/clamd'
AVENGINE_CONF='/etc/clamav/clamd.conf'
PROSCAN_CONF='/etc/opt/proscan/proscan.conf'
WGET=`which wget`
CLAMAV_ENGINE_IDX='/tmp/.psupd_engine.idx'
CLAMAV_ENGINE_VER='/tmp/.psupd_engine.ver'
PROSCAN=/opt/proscan/bin/proscan

if [ ! -x "$UPDATER" ]
then
	echo ''
	echo "Cannot execute $UPDATER"
	exit 2
fi

if [ "$WGET" = "" -o ! -x "$WGET" ]
then
	echo ""
	echo "wget program not found"
	exit 2
fi

CONFIG="--config-file=$UPDATER_CONF"
AVUPDATE_PARAMETER="--stdout --no-dns --no-warnings"
UPDATE_ERR=0
CONFIG_RESET=0
UPDATE_PORT=80

for PARAM in "$@"
do
	case ${PARAM} in
		--config-file=*)
			CONF_LOCATION=`echo $PARAM | cut -d '=' -f2`
			CONFIG=$PARAM
			UPDATER_CONF=$CONF_LOCATION
			CONFIG_RESET=1
			;;

		--update-host=*)
			UPDATE_HOST=`echo $PARAM | cut -d '=' -f2`
			if [ "$UPDATE_HOST" = "" ]
			then
				exit 2
			fi
			;;

		--update-port=*)
			UPDATE_PORT=`echo $PARAM | cut -d '=' -f2`
			if [ "$UPDATE_PORT" = "" ]
			then
				UPDATE_PORT=80
			fi
			;;

		-V|--version)
			$UPDATER --version
			exit 0
			;;

		-h|--help)
			$UPDATER --help
			exit 0
			;;

		-b)
			CHECK_PARAM=$2
			if [ "$CHECK_PARAM" = "check" ]
			then
			    exit 2
			fi
			if [ $CHECK_PARAM -gt $VER ]
			then
			    exit 1
			fi
			exit 0
			;;

		*)
			echo "unkown script parameter found. the parameter will be used for direct freshclam start ... "
			AVUPDATE_PARAMETER="${PARAM} ${AVUPDATE_PARAMETER}"
			;;
	esac
done

if [ $CONFIG_RESET -eq 0 ]
then
	# config setup
	echo "#--- freshclam.conf ----
DatabaseDirectory /usr/lib/clamav" > $UPDATER_CONF

	LOG_VERBOSE=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^LogVerbose/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$LOG_VERBOSE" != "" ]
	then
	    echo "LogVerbose $LOG_VERBOSE" >> $UPDATER_CONF
	fi

	if [ "$UPDATE_HOST" = "" ]
	then
		ENG_UPD_HOST=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^EngineUpdateServer/{print $2;flag=0;}|cut -d, -f1' $PROSCAN_CONF`
		if [ "$ENG_UPD_HOST" != "" ]
		then
		    UPDATE_HOST=$ENG_UPD_HOST
		else
		    UPDATE_HOST=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^UpdateHost/{print $2;flag=0;}|cut -d, -f1' $PROSCAN_CONF`
		fi
	fi

	if [ "$UPDATE_PORT" = "" ]
	then
		ENG_UPD_PORT=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^EngineUpdatePort/{print $2;flag=0;}' $PROSCAN_CONF`
		if [ "$ENG_UPD_PORT" != "" ]
		then
		    UPDATE_PORT=$ENG_UPD_PORT
		fi
	fi

	ENG_UPD_PROTOCOL=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^UpdateProtocol/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$ENG_UPD_PROTOCOL" != "" ]
	then
	    UPDATE_PROTOCOL=`echo $ENG_UPD_PROTOCOL|tr '[:upper:]' '[:lower:]'`
	fi

	echo "PrivateMirror $UPDATE_HOST" >> $UPDATER_CONF

	SCRIPTED_UPDATES=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^ScriptedUpdates/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$SCRIPTED_UPDATES" != "" ]
	then
	    echo "ScriptedUpdates $SCRIPTED_UPDATES" >> $UPDATER_CONF
	else
	    echo "ScriptedUpdates no" >> $UPDATER_CONF
	fi

	UPDATE_DNS=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^DNSDatabaseInfo/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$UPDATE_DNS" != "" ]
	then
	    echo "DNSDatabaseInfo $UPDATE_DNS" >> $UPDATER_CONF
	fi

	UPDATE_LOGFILE=`awk -F= 'BEGIN{flag=0;}/^\[updater.report\]/{flag=1;next;}flag==1&&/^ReportFileName/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$UPDATE_LOGFILE" != "" ]
	then
	    if [ "$UPDATE_LOGFILE" = "syslog" ]
	    then
		echo "LogSyslog yes" >> $UPDATER_CONF
		LOG_FACILITY=`awk -F= 'BEGIN{flag=0;}/^\[updater.report\]/{flag=1;next;}flag==1&&/^Facility/{print $2;flag=0;}' $PROSCAN_CONF`
		conv_facility $LOG_FACILITY
		echo "LogFacility $FACILITY" >> $UPDATER_CONF
	    else
		echo "UpdateLogFile $UPDATE_LOGFILE" >> $UPDATER_CONF
	    fi
	fi
	EXEC_USER=`awk -F= 'BEGIN{flag=0;}/^\[aveserver\]/{flag=1;next;}flag==1&&/^ExecUser/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$EXEC_USER" != "" ]
	then
	    echo "DatabaseOwner $EXEC_USER" >> $UPDATER_CONF
	fi
	HTTP_PROXY_HOST=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^HTTPProxyServer/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$HTTP_PROXY_HOST" != "" ]
	then
	    echo "HTTPProxyServer $HTTP_PROXY_HOST" >> $UPDATER_CONF
	fi
	HTTP_PROXY_PORT=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^HTTPProxyPort/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$HTTP_PROXY_PORT" != "" ]
	then
	    echo "HTTPProxyPort $HTTP_PROXY_PORT" >> $UPDATER_CONF
	fi
	HTTP_PROXY_USER=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^HTTPProxyUser/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$HTTP_PROXY_USER" != "" ]
	then
	    echo "HTTPProxyUsername $HTTP_PROXY_USER" >> $UPDATER_CONF
	fi
	HTTP_PROXY_PASS=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^HTTPProxyPassword/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$HTTP_PROXY_PASS" != "" ]
	then
	    echo "HTTPProxyPassword $HTTP_PROXY_PASS" >> $UPDATER_CONF
	fi
	RETRY_TIME=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^RetryTimes/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$RETRY_TIME" != "" ]
	then
	    echo "MaxAttempts $RETRY_TIME" >> $UPDATER_CONF
	fi
	RETRY_INTERVAL=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^RetryInterval/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$RETRY_INTERVAL" != "" ]
	then
	    echo "ConnectTimeout $RETRY_INTERVAL" >> $UPDATER_CONF
	fi
	APPEND_FILE=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^AppendFile/{print $2;flag=0;}' $PROSCAN_CONF`
	if [ "$APPEND_FILE" != "" ]
	then
	    cat $APPEND_FILE >> $UPDATER_CONF
	fi
	RELOAD_APPL=`awk -F= 'BEGIN{flag=0;}/^\[updater.options\]/{flag=1;next;}flag==1&&/^ReloadApplication/{print $2;flag=0;}' $PROSCAN_CONF`
fi

echo "Updating, please wait..."
echo

# proxy setting for enveroment
if [ "$HTTP_PROXY_HOST" != "" ]
then
    http_proxy="$UPDATE_PROTOCOL://$HTTP_PROXY_HOST:$HTTP_PROXY_PORT"
    export http_proxy
fi
if [ "$HTTP_PROXY_USER" != "" ]
then
    proxy_user="$HTTP_PROXY_USER"
    export proxy_user
fi
if [ "$HTTP_PROXY_PASS" != "" ]
then
    proxy_password="$HTTP_PROXY_PASS"
    export proxy_password
fi

#----------------------------------
if [ "$MODE" != "test" ]
then

	# clamd update
	$WGET -q -O $CLAMAV_ENGINE_VER $UPDATE_PROTOCOL://$UPDATE_HOST:$UPDATE_PORT/clamav/idx/engine.ver

	# check index file
	if [ ! -d $CLAMAV_LIB/.update_engine ]
	then
		mkdir $CLAMAV_LIB/.update_engine
	else
		rm -f $CLAMAV_LIB/.update_engine/* > /dev/null
	fi
	
	UPDATE=0
	DOWNLOAD=0
	ENGINE_DOWNLOAD=0
	
	if [ -s $CLAMAV_ENGINE_VER ]
	then
	
		NEWVER=`awk -F. '{printf("%02d%02d%02d",$1,$2,$3)}' $CLAMAV_ENGINE_VER`
		OLDVER=`$CLAMAV_LIB/sigtool -V|cut -d '/' -f 1|sed -e 's/ClamAV //'|awk -F. '{printf("%02d%02d%02d",$1,$2,$3)}'`
		
		if [ "$NEWVER" = "" ]
		then
		    NEWVER=0
		fi
		if [ "$OLDVER" = "" ]
		then
		    OLDVER=0
		fi
		
		if [ $NEWVER -gt $OLDVER ]
		then
		
			$WGET -q -O $CLAMAV_ENGINE_IDX $UPDATE_PROTOCOL://$UPDATE_HOST:$UPDATE_PORT/clamav/idx/engine.idx
			
			while read MD5 FILE
			do
				if [ -f $CLAMAV_LIB/$FILE ]
				then
					CUR=`md5sum $CLAMAV_LIB/$FILE | cut -d ' ' -f1`
				else
					CUR=""
				fi
				if [ "$CUR" != "$MD5" ]
				then
					# download 
					echo "New ClamAV module found($FILE,$MD5)"
					$WGET -q -O $CLAMAV_LIB/.update_engine/$FILE.gz $UPDATE_PROTOCOL://$UPDATE_HOST:$UPDATE_PORT/clamav/engine/$FILE.gz
					if [ -f $CLAMAV_LIB/.update_engine/$FILE.gz ]
					then
						echo "... downloding complete. extracting ..."
						(cd $CLAMAV_LIB/.update_engine;gunzip $FILE.gz;chmod a+x $FILE)
						if [ "$FILE" = "clamd" ]
						then
							ENGINE_DOWNLOAD=1
						fi
						DOWNLOAD=1
					else
						echo "... downloading failed"
						echo
						echo "Update failed"
						exit 1
					fi
				fi
			done < $CLAMAV_ENGINE_IDX
			
		else
			echo "current version($OLDVER) is latest version"
		fi
	
	else
		echo "no engine index or version file"
	fi

	# pretest & install
	if [ $ENGINE_DOWNLOAD -eq 1 -a "$RELOAD_APPL" = "yes" ]
	then
		PUSH_CWD=`pwd`
		cd $CLAMAV_LIB/.update_engine

		# Pre TEST
		if [ "$OS" = "linux" ]
		then

			LIBS=`ls lib* 2> /dev/null`
			if [ "$LIBS" != "" ]
			then
				for NEWLIB in $LIBS
				do
					PRELOAD="$PRELOAD:./$NEWLIB"
				done
			fi
			if [ "$PRELOAD" != "" ]
			then
				export LD_PRELOAD=$PRELOAD
			fi
			if [ ! -x clamd ]
			then
				cp -p $AVENGINE .
			fi

		elif [ "$OS" = "aix" ]
		then

			PUSH_LIBPATH=$LIBPATH
			LIBPATH=.
			if [ ! -x clamd ]
			then
				cp -p $AVENGINE .
			fi
			echo "pretest engine version : \c"

		else
			exit 1
		fi

		# version check
		./clamd -V
		if [ $? -ne 0 ]
		then
			echo "Engine pretest failed"
			echo
			echo "Update failed"

		else
			export LIBPATH=$PUSH_LIBPATH

			# New Engine Apply
			NEW_VERSION=`./clamd -V`
			export LD_PRELOAD=
			OLD_VERSION=`$AVENGINE -V`
			cp -f * $CLAMAV_LIB
			if [ "$OS" = "linux" ]
			then
				sh $CLAMAV_LIB/post_install.sh
			fi
			$PROSCAN stop
			sleep 2
			$PROSCAN
			if [ $? -eq 0 ]
			then
				echo "Reloading Engine successfull"
				echo "$OLD_VERSION --> $NEW_VERSION"
				UPDATE=1
			else
				echo "Engine restart failed"
				echo
				echo "Update failed"
				exit 1
			fi
		fi

		cd $PUSH_CWD
		echo

	elif [ $DOWNLOAD -eq 1 -a "$RELOAD_APPL" = "yes" ]
	then
		PUSH_CWD=`pwd`
		cd $CLAMAV_LIB/.update_engine
		cp -f * $CLAMAV_LIB
		echo "Reloading ClamAV module successfull"
		echo
		cd $PUSH_CWD
		UPDATE=1
	fi
fi

#----------------------------------

# VDF update
LOGOUT=/tmp/freshclam_console.log
$UPDATER $CONFIG ${AVUPDATE_PARAMETER} | grep -v "Downloading" | tee $LOGOUT
UPDATE_ERR="$?"

DOWNLOAD=`grep "Database test passed" $LOGOUT`
ERROR=`grep -i "failed|error|fatalv" $LOGOUT`

rm $LOGOUT

if [ $UPDATE_ERR -eq 0 -a "$DOWNLOAD" != "" ]
then
    if [ -f "$AVENGINE_CONF" ]
    then
	PID_FILE=`awk '/PidFile /{print $2}' $AVENGINE_CONF`
    fi
    if [ -f "$PID_FILE" ]
    then
	PID=`cat $PID_FILE`
	CHK=`ps ax | grep "$AVENGINE" | grep "$PID" | grep -v grep`
	if [ "$CHK" != "" ]
	then
	    echo
	    echo "Reloading Database ..."
	    kill -USR2 $PID
	else
	    rm $PID_FILE
	    SOCKET_FILE=`awk '/LocalSocket /{print $2}' $AVENGINE_CONF`
	    if [ -S "$SOCKET_FILE" ]
	    then
		rm $SOCKET_FILE
	    fi
	fi
    fi
    echo
    echo "Virus Database update successfull"
elif [ $UPDATE_ERR -eq 0 -a "$ERROR" = "" ]
then
    echo
    echo "Nothing to Database update"
else
    echo
    echo "Update failed"
fi

exit "$UPDATE_ERR"
